<?php
// inteq_notes.php - Handles Locked Notes Webhook (using PHPMailer for simplicity)
header('Content-Type: application/json');

// Include PHPMailer (via Composer)
require_once __DIR__ . '/vendor/autoload.php';
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

// Custom debug log file
define('DEBUG_LOG_FILE', __DIR__ . '/debug.log');

// Constants (exact match to original)
define('WEBHOOK_LOG_FILE_FOR_NOTES', __DIR__ . '/notes_webhook_logs.json');

// IntakeQ API
define('INTAKEQ_API_KEY', '45eaf2f1c5ba25e9a4cebfb040686d910152149c');

// Email config
define('EMAIL_HOST', 'smtp.gmail.com');
define('EMAIL_PORT', 587);
define('EMAIL_USE_TLS', true);
define('EMAIL_HOST_USER', 'admin@aderishealth.com');
define('EMAIL_HOST_PASSWORD', 'pqxqdpmvhwqfkxmj');
define('DEFAULT_FROM_EMAIL', 'Medscape <admin@aderishealth.com>');

define('ADMIN_EMAIL', 'admin@aderishealth.com');   // change if needed

define('NOTES_DIR', __DIR__ . '/notes-form');  // Full path for safety

// NOTE_EMAILS (exact match to original)
$NOTE_EMAILS = [
    'California Aesthetics Note (Milpitas)' => 'kathleen@californiaaesthetics.co, greg@californiaaesthetics.co, laa@californiaaesthetics.co, alissa@californiaaesthetics.co',
    'California Aesthetic Note (Glendale)' => 'dawn@californiaaesthetics.co, karla@californiaaesthetics.co, laa@californiaaesthetics.co, maranda@californiaaesthetics.co, anna@californiaaesthetics.co',
    'California Aesthetic Note (Orange)' => 'megan@californiaaesthetics.co, lidia@californiaaesthetics.co, laa@californiaaesthetics.co, tara@californiaaesthetics.co',
    'Diamond\'s Unique Client Note' => '',
    'Extravagant Pampering Note' => 'extravagantpampering@yahoo.com',
    'Impact Body Worx Note' => 'info@impactbodyworx.com',
    'My Ageless Lounge Note' => 'christa@skinbychrista.com',
    'Reclaim Health Note' => 'info@reclaim-nc.com',
    'Diamond\'s Unique Wellness Note' => 'diamonds uniquewellness@gmail.com',
    'Regenesis Wellness Notes' => 'riley@regenesis-wellness.com',
    'Regen Therapeutics Atlanta Note' => 'nsprphg@gmail.com',
    'Skynn Medical Aesthetics Note' => 'briannefaas123@gmail.com',
    'NEW SevenReveries Note' => '',
    'TinyTox Collab Note' => 'tinytoxcollab@gmail.com, admin@aderishealth.com',
    'Replenish Health Spa Note' => 'Sam@replenishhealthspa.com',
    'Dr. Danilevsky Aesthetic Medicine Note' => 'endorphinmedcorp@gmail.com',
    'SevenReveries Note' => 'ale.tkachenko@gmail.com',
    'Sage Revive Note' => 'rjmalhotra@gmail.com, ashleyholliday@sagerevivemoorestown.com',
    'Sample Note For TESTING' => 'snakoti511@gmail.com, asddfff@yopmail.com',
    'Sage Revive Documentation Note' => '',
    'My Thera Touch Note' => 'theratouchandhealing@gmail.com',
    'AcneClinicNYC Note' => 'acneclinicnyc1@gmail.com',
    'Modern Cryo & Wellness Note' => 'chris@moderncryo.com',
    'Lashes By Ty LLC Note' => 'lashesbyty@icloud.com',
    'Arizona Aesthetics Note' => '',
    'NovaStem Note' => '',
    'Nature\'s Drip Note' => 'ashatthecornucopia@gmail.com',
    'A Client Note' => '',
    'Exosomes Hair Restoration' => '',
    'Transformation Station Note' => 'resultstremellc@gmail.com',
    'Reviva Aesthetics Note' => 'contact@reviva-aesthetics.com, mya@reviva-aesthetics.com, contactrevivaaesthetics@gmail.com',
    'New York Beauty Center Note' => 'newyorkbeautycenter@gmail.com',
    'Inspired Beauty Med Spa Note' => 'info@inspiredbeauty.boutique',
    'The Protocole Note' => 'cindy@theprotocole.com, delphine@theprotocole.com'
];

// Helper: Custom debug log to file
function debug_log($message) {
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] DEBUG (inteq_notes.php): $message" . PHP_EOL;
    file_put_contents(DEBUG_LOG_FILE, $log_entry, FILE_APPEND | LOCK_EX);
}

// Helper: Send email with attachments (for notes) using PHPMailer
function send_notes_email($subject, $body, $recipient_list, $attachments = null) {
    debug_log("send_notes_email called with subject: $subject, recipients: " . implode(', ', $recipient_list) . ", attachments: " . json_encode($attachments));
    $mail = new PHPMailer(true);
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = EMAIL_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = EMAIL_HOST_USER;
        $mail->Password = EMAIL_HOST_PASSWORD;
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port = EMAIL_PORT;

        // Sender and recipients
        $mail->setFrom(EMAIL_HOST_USER, 'Medscape');
        foreach ($recipient_list as $recipient) {
            $mail->addAddress($recipient);
        }

        // Content
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $body;

        // Attachments
        if ($attachments) {
            foreach ($attachments as $file_path) {
                debug_log("Processing attachment: $file_path");
                if (file_exists($file_path)) {
                    $mail->addAttachment($file_path);
                    debug_log("Attachment added: " . basename($file_path));
                } else {
                    debug_log("Attachment missing: $file_path");
                }
            }
        }

        $mail->send();
        debug_log("Email sent success");
        return true;
    } catch (Exception $e) {
        debug_log("Email failed: " . $mail->ErrorInfo);
        return false;
    }
}

// Helper: Log to JSON file - with explicit check
function log_to_json($log_file, $payload) {
    debug_log("log_to_json called for file: $log_file, payload: " . json_encode($payload));
    $event_data = [
        'timestamp' => date('c'), // ISO format equivalent to now().isoformat()
        'data' => $payload
    ];
    $logs = [];
    if (file_exists($log_file)) {
        $log_content = file_get_contents($log_file);
        if ($log_content !== false) {
            $logs = json_decode($log_content, true) ?: [];
            debug_log("Loaded existing logs, count: " . count($logs));
        } else {
            debug_log("file_get_contents failed for $log_file");
        }
    } else {
        debug_log("Log file $log_file does not exist, creating new");
    }
    $logs[] = $event_data;
    $write_result = file_put_contents($log_file, json_encode($logs, JSON_PRETTY_PRINT));
    if ($write_result === false) {
        debug_log("CRITICAL: file_put_contents FAILED for $log_file - check permissions!");
    } else {
        debug_log("Log written successfully, bytes: $write_result");
    }
}

// Helper: Make API request with cURL (equivalent to requests.get) - with debug
function api_get($url, $headers) {
    debug_log("api_get called for URL: $url");
    if (!function_exists('curl_init')) {
        debug_log("CRITICAL: cURL not enabled!");
        return ['status_code' => 0, 'body' => 'cURL missing'];
    }
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // For testing; enable in prod
    $response = curl_exec($ch);
    $status_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    if ($curl_error) {
        debug_log("cURL error: $curl_error");
    }
    debug_log("API response status: $status_code, first 100 chars: " . substr($response, 0, 100));
    return ['status_code' => $status_code, 'body' => $response];
}

// Main Logic for inteq_notes.php
debug_log("Script started, method: " . ($_SERVER['REQUEST_METHOD'] ?? 'NONE'));
$method = $_SERVER['REQUEST_METHOD'] ?? '';

if ($method !== 'POST') {
    debug_log("Non-POST method: $method");
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Only POST allowed']);
    exit;
}

debug_log("POST detected, reading input");
$input = file_get_contents('php://input');
debug_log("Raw input length: " . strlen($input ?? ''));

$payload = json_decode($input, true);
$json_error = json_last_error();
if ($json_error !== JSON_ERROR_NONE) {
    debug_log("CRITICAL: JSON decode failed: " . json_last_error_msg());
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => json_last_error_msg()]);
    exit;
}

debug_log("Payload parsed: " . json_encode($payload));

// Log webhook - always, even for wrong Type
log_to_json(WEBHOOK_LOG_FILE_FOR_NOTES, $payload);

// Process if Note Locked
$type = $payload['Type'] ?? 'MISSING';
debug_log("Payload Type: $type");
if ($type !== 'Note Locked') {
    debug_log("Type not 'Note Locked', skipping processing but logged");
    echo json_encode(['status' => 'success', 'message' => 'Webhook processed successfully (no action)']);
    exit;
}

debug_log("Type is 'Note Locked', entering try");
try {
    $note_id = $payload['NoteId'] ?? null;
    debug_log("NoteId: " . ($note_id ?? 'MISSING'));
    if (!$note_id) {
        throw new Exception('No NoteId in payload');
    }

    $pdf_url = 'https://intakeq.com/api/v1/notes/' . $note_id . '/pdf';
    $headers = ['X-Auth-Key: ' . INTAKEQ_API_KEY];
    $default_note_url = 'https://intakeq.com/api/v1/notes/' . $note_id;

    // Step 1: Get Note details
    debug_log("Fetching note details from $default_note_url");
    $default_res = api_get($default_note_url, $headers);
    if ($default_res['status_code'] !== 200) {
        throw new Exception('Failed to fetch note details: ' . $default_res['body']);
    }
    $data = json_decode($default_res['body'], true);
    $note_name = trim($data['NoteName'] ?? '');
    $client_name = trim($data['ClientName'] ?? '');
    debug_log("Note name: $note_name, Client: $client_name");

    // Step 2: Download PDF
    debug_log("Downloading PDF from $pdf_url");
    $pdf_res = api_get($pdf_url, $headers);
    if ($pdf_res['status_code'] !== 200) {
        throw new Exception('Failed to download PDF: ' . $pdf_res['body']);
    }
    debug_log("PDF downloaded, size: " . strlen($pdf_res['body']));

    // Sanitize note_name
    $safe_note_name = preg_replace('/[^a-zA-Z0-9 _-]/', '_', $note_name);
    debug_log("Safe note name: $safe_note_name");

    // Create subdir
    $note_dir = NOTES_DIR . '/' . $safe_note_name;
    debug_log("Full note dir: $note_dir");
    if (!is_dir($note_dir)) {
        debug_log("Creating subdir $note_dir");
        $mkdir_result = mkdir($note_dir, 0755, true);
        if (!$mkdir_result) {
            throw new Exception('Failed to create subdir: ' . $note_dir . ' - check permissions');
        }
        debug_log("Subdir created");
    } else {
        debug_log("Subdir exists");
    }

    $file_path = $note_dir . '/' . $note_id . '.pdf';
    debug_log("Saving PDF to $file_path");
    $write_result = file_put_contents($file_path, $pdf_res['body']);
    if ($write_result === false) {
        throw new Exception('Failed to save PDF: ' . $file_path . ' - check permissions');
    }
    debug_log("PDF saved, bytes: $write_result");

    // Common message body
    $message_body = <<<HTML
<!DOCTYPE html>
<html>
<body>
    <p>Dear Recipient,</p>
    <p>Please be advised that the note for <b>{$client_name}</b> has been signed.</p>
    <p>Please see the attached PDF.</p>
    <p>Best regards,</p>
    <p>MedScape GFE</p>
</body>
</html>
HTML;
    debug_log("Message body prepared");

    // Send to Clinic (if exists)
    $normalized_note_emails = [];
    foreach ($NOTE_EMAILS as $k => $v) {
        $normalized_note_emails[strtolower($k)] = $v;
    }
    $clinic_email_str = $normalized_note_emails[strtolower($note_name)] ?? '';
    debug_log("Clinic email str: $clinic_email_str");
    if ($clinic_email_str) {
        $clinic_recipients = array_filter(array_map('trim', explode(',', $clinic_email_str)));
        $clinic_recipient_list = array_unique(array_merge($clinic_recipients, [ADMIN_EMAIL]));
        debug_log("Clinic recipients: " . implode(', ', $clinic_recipient_list));
        if (!empty($clinic_recipient_list)) {
            $subject = 'New Locked Note - ' . ($note_name ?: $note_id) . ' by ' . $client_name;
            debug_log("Calling send_notes_email");
            $email_sent = send_notes_email($subject, $message_body, $clinic_recipient_list, [$file_path]);
            debug_log("Notes email sent: " . ($email_sent ? 'YES' : 'NO'));
        } else {
            debug_log("No valid recipients after filter");
        }
    } else {
        debug_log("No clinic match for note name: $note_name");
    }

    debug_log("All steps complete, returning success");
    echo json_encode(['status' => 'success', 'message' => 'Webhook processed successfully']);
} catch (Exception $e) {
    debug_log('Exception in try: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
?>